/*
 * Copyright (C) 1997-2000 SpellCaster Telecommunications Inc.
 * $Id: loadtest.c,v 1.1.1.1 1999/12/30 15:59:24 mark Exp $
 * Released under the GNU Public License. See LICENSE file for details.
 */

#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <errno.h>

#include "aps_if.h"

long tx_bytes[256], rx_bytes[256];
int fds[256];
char devname[256];

#define MULT 12

int nchan = 192;
int dialout = 0, doing_dialout = 0;
int pktlen = 540;

void do_open(int idx)
{
	char buf[32];
	int ret;

	sprintf(devname, "/dev/bab/%d", idx+0 + dialout+(idx/23));

	fds[idx] = open(devname, O_RDWR | (doing_dialout ? O_NONBLOCK : 0));
	if (-1 == fds[idx]) {
		fprintf(stderr, "Error opening %s: %s\n", devname, strerror(errno));
		exit(1);
	}

	if (doing_dialout) {
		int fl = fcntl(fds[idx], F_GETFL, 0);
		fl &= ~O_NONBLOCK;
		fcntl(fds[idx], F_SETFL, fl);

		sprintf(buf, "811%d%02d", (idx/23)+1, idx%23);
		printf("Dialing %s...\n", buf);
		if ((ret = ioctl(fds[idx], BIOCDIAL, buf)) < 0) {
			perror("Dial failed");
			exit(1);
		}

		if (ret > 0) {
			fprintf(stderr, "Dial failed cause 0x%02x\n", ret);
			exit(1);
		}
	}
	printf("%s connected\n", devname);
}

int main ( int argc, char *argv[] )
{
	char buf[4096];
	fd_set rfds, wfds;
	struct timeval tv, stv, ctv;
	long next_sec;
	int i;

	if (argc < 3) {
		fprintf(stderr, "usage: %s <packet len> <num chan> <dialout>\n", argv[0]);
		return 1;
	}

	pktlen = atoi(argv[1]);
	nchan = atoi(argv[2]);

	doing_dialout = (argc > 3);
	dialout = doing_dialout ? atoi(argv[3]) : 0;

	for (i=0; i<nchan; i++)
		do_open(i);

	FD_ZERO(&rfds);
	FD_ZERO(&wfds);

	gettimeofday(&stv, NULL);
	next_sec = stv.tv_sec + 3;
	tv.tv_sec = 3;
	tv.tv_usec = 0;

	for (;;) {
		for (i=0; i<nchan; i++) {
			FD_SET(fds[i], &rfds);
			FD_SET(fds[i], &wfds);
		}

		gettimeofday(&ctv, NULL);
		if ((ctv.tv_sec > next_sec) || (ctv.tv_sec == next_sec && ctv.tv_usec >= stv.tv_usec))
			goto timeout;

		tv.tv_sec = next_sec - ctv.tv_sec;
		tv.tv_usec = stv.tv_usec - ctv.tv_usec;
		if (tv.tv_usec < 0) {
			tv.tv_usec += 1000000;
			tv.tv_sec --;
		}

		if (-1 == (i = select(fds[nchan-1]+1, &rfds, &wfds, NULL, &tv))) {
			perror("select");
			exit(1);
		}

		if (!i) {
			long total;

timeout:
			gettimeofday(&tv, NULL);
			next_sec = tv.tv_sec + 3;
			total = (tv.tv_sec - stv.tv_sec) * 100 + (tv.tv_usec - stv.tv_usec) / 10000;

			printf("\ntime %ld.%02ld\n\n", total/100, total%100);
			for (i=0; i<nchan; i++) {
				printf("%2d:  %10ld  %10ld  %5ld/s  %5ld/s\n",
					i,
					tx_bytes[i], rx_bytes[i],
					(tx_bytes[i] / ((total+99) / 100)),
					(rx_bytes[i] / ((total+99) / 100))
					);
			}

			tv.tv_sec = 3;
			tv.tv_usec = 0;
			continue;
		}

		for (i=0; i<nchan; i++) {
			if (FD_ISSET(fds[i], &rfds)) {
				int len = read(fds[i], &buf, sizeof(buf));
				if (len <= 0) {
					perror("read");
					exit(1);
				}
				rx_bytes[i] += len;
			}

			if (FD_ISSET(fds[i], &wfds)) {
				int len = write(fds[i], &buf, pktlen);
				if (len != pktlen) {
					perror("read");
					exit(1);
				}
				tx_bytes[i] += len;
			}
		}
	}
}

