/*
 * Copyright (C) 1997-2000 SpellCaster Telecommunications Inc.
 * $Id: timer.h,v 1.3 2004/08/14 23:23:04 bcrl Exp $
 * Released under the GNU Public License. See LICENSE file for details.
 */


#ifndef CTIMER_H
#define CTIMER_H

#include <sys/time.h>
#include <unistd.h>
#include <stdio.h>

class TimeVal {
public:
	struct timeval	tv;

	TimeVal()
	{
		tv.tv_sec = 0;
		tv.tv_usec = 0;
	}

	long long scalar(void)
	{
		return tv.tv_sec * 1000000LL + tv.tv_usec;
	}

	void Normalize(void)
	{
		long long val = scalar();
		tv.tv_sec = val / 1000000;
		tv.tv_usec = val % 1000000;
	}

	void AddHsecs(long hsecs)
	{
		tv.tv_sec += hsecs / 100;
		tv.tv_usec += (hsecs % 100) * (1000000 / 100);
		Normalize();
	}

	void AddSecs(long secs)
	{
		tv.tv_sec += secs;
	}

	void GetTimeOfDay(void)
	{
		gettimeofday(&tv, NULL);
	}

	TimeVal operator+(TimeVal r)
	{
		TimeVal res;
		res.tv.tv_sec = tv.tv_sec + r.tv.tv_sec;
		res.tv.tv_usec = tv.tv_usec + r.tv.tv_usec;
		res.Normalize();
		return res;
	}

	TimeVal operator-(TimeVal r)
	{
		TimeVal res;
		res.tv.tv_sec = tv.tv_sec - r.tv.tv_sec;
		res.tv.tv_usec = tv.tv_usec - r.tv.tv_usec;
		res.Normalize();
		return res;
	}

	int operator<(TimeVal r)
	{
		r = r - *this;
		return (r.tv.tv_sec > 0 || r.tv.tv_usec > 0);
	}

	int operator<=(TimeVal r)
	{
		r = r - *this;
		return (r.tv.tv_sec >= 0 && r.tv.tv_usec >= 0);
	}
};

extern void timer_run(void);
extern struct timeval timer_getdelay(void);

class CTimer {
protected:
	CTimer	*next, *prev;
	CTimer	**my_timer_list;

	TimeVal	etv;
	unsigned	set : 1, active : 1;

	void	(*cbf)(void *cbd);
	void	*cbd;

public:
	CTimer(unsigned long hsecs = 0)
	{
		set = 0;
		active = 0;
		cbf = NULL;
		cbd = NULL;

		next = prev = NULL;
		my_timer_list = NULL;

		if (hsecs)
			Start(hsecs);
	}

	virtual ~CTimer() {
		if (next || prev)
			remove_from_timer_list();
	}

	void insert_into_timer_list(void);
	void remove_from_timer_list(void);

	friend void dump_timer_list(int i, CTimer *timer_list);
	friend void run_timer_list(TimeVal *, CTimer **);
	friend struct timeval timer_getdelay(void);

	CTimer(void (*func)(void *), void *data)
	{
		CTimer();
		SetFunction(func, data);
	}

	void Stop(void)
	{
		active = 0;
		remove_from_timer_list();
	}

	void Start(void)
	{
		if (set) {
			active = 1;
			insert_into_timer_list();
#ifdef DEBUG
			fprintf(stderr, "Start timer(%p): expires %d.%06d callback: %p(%p)\n",
				this, etv.tv.tv_sec, etv.tv.tv_usec, cbf, cbd);
#endif
		}
	}

	void Start(unsigned long hsecs)
	{
		Set(hsecs);
		Start();
	}

	void SetFunction(void (*func)(void *), void *data)
	{
		cbf = func;
		cbd = data;
	}

	void Set(unsigned long hsecs)
	{
		set = 1;
		etv.GetTimeOfDay();
		etv.AddHsecs(hsecs);
	}

	int IsActive(void)
	{
		return active;
	}

	virtual void TimerExpired(void);
};


#endif

