/*
 * main.cc - Self explanitory
 * Copyright (C) 1997-2000 SpellCaster Telecommunications Inc.
 * $Id: main.cc,v 1.1.1.1 2004/03/11 03:59:31 bcrl Exp $
 * Released under the GNU Public License. See LICENSE file for details.
 */

#include <stdlib.h>
#include <time.h>

#include "kernel.h"
#include "debug.h"
#include "protocol.h"
#include "config.h"
#include "babd.h"


static char version[] = VER;

struct aps_idev_class;

/* Internal Device Class Structure */
typedef struct aps_idev_class {
  struct aps_idev_class *prev;
	unsigned int id;
	char dev_name[10];
} aps_idev_class_t;


static aps_idev_class_t *devClassList = NULL;

extern unsigned int GetEthAddress(u8 *);

extern char *sysname;

/* Debugging flags */
extern unsigned short log_flags;
extern unsigned char log_facility;

/* User settable debug level flag */
long debug = 0;

/*
 * Global default link config policy
 */
policy_t g_policy;

extern int aps_init();
extern void aps_cleanup();
extern unsigned int RegisterDeviceClass(char *);
extern void UnregisterDeviceClass(unsigned int);

int aps_init() 
{
	u8 epd_class;
	u8 epd_addr[20];
	u8 epd_length;

	dbg_ptr->log_flags = LF_ALL|(debug * 0x1000);
	dbg_ptr->log_facility = LT_STDERR;
	InitLog("babylon");

	printf("Babylon version %s\n", version);

	/*
	 * Establish our endpoint descriptor
	 */
#if 0
	epd_class = 3;
	epd_length = GetEthAddress(epd_addr);
	if(epd_length == 0)
#endif
	{
		/*
		 * Create an epd from a magic number block * 6
		 */
		srandom(time(NULL));
		*((u32 *)epd_addr) = random() ^ (u32)&g_policy;
		*(2+(u16 *)epd_addr) = random() ^ (u32)&g_policy;
		epd_length = 6;
		epd_class = 3;
	}
	 
	/*
	 * Set our initail system name
	 */
	if (!sysname)
		sysname = "babylon";

	/*
	 * Setup the global link policy
	 */
	memset(&g_policy, 0, sizeof(policy_t));
	// Let these be negotiated
	g_policy.Set(P_MRU, 1500);
	//g_policy.Set(P_MRRU, 1500);
	g_policy.Set(P_SSN, 1);
	g_policy.Set(P_PFC, 1);
	g_policy.Set(P_ACFC, 1);
	g_policy.Set(P_AUTH, PPP_PROTO_PAP);
	g_policy.Set(P_ANSWER, 1);
	g_policy.Set(P_ECHO, 10);
	g_policy.Set(P_EPD_CLASS, epd_class);
	g_policy.Set(P_EPD_LENGTH, epd_length);
	memcpy(&g_policy.epd_addr, &epd_addr, epd_length);

	return 0;
}

void aps_cleanup() 
{
	aps_idev_class_t *pDev = devClassList, *pLst;

	while(pDev != NULL) {
		pLst = pDev->prev;
		delete pDev;
		pDev = pLst;
	}
		
	printf("SpellCaster APS Protocol Driver unloaded\n");
}


/* Register a new device class driver with aps */
unsigned int RegisterDeviceClass(char *name) 
{
	aps_idev_class_t *pDev = new aps_idev_class_t;

	DebugEnter("RegisterDeviceClass()");

	if(pDev == NULL) {
		Log(LF_ERROR|LF_ALLOC, "Out of memory registering device class");
		DebugReturn(0);
	}

	pDev->prev = devClassList;
	devClassList = pDev;
	pDev->id = (unsigned int)name;
	strcpy(pDev->dev_name, name);

	DebugReturn(pDev->id);
}


/* Unregister Device Class */
void UnregisterDeviceClass(unsigned int id) 
{
	aps_idev_class_t *pNxt = NULL, *pDel = devClassList;

	DebugEnter("UnregisterDeviceClass()");

	while((pDel != NULL) && (pDel->id != id)) {
		pNxt = pDel;
		pDel = pDel->prev;
	}

	Log(LF_DEBUG|LF_LIST, "pDel %p, pNxt %p, prev %p", pDel, pNxt, pDel->prev);
	if(pDel == NULL) {
		Log(LF_DEBUG|LF_LIST, "Invalid DeviceClass id(0x%x)", id);	
		DebugVoidReturn;
	}

	if(pNxt != NULL) 
		pNxt->prev = pDel->prev;

	if(devClassList == pDel) {
		Log(LF_DEBUG|LF_LIST, "Removed last deviceclass");
		devClassList = NULL;
	}

	delete pDel;
	DebugExit();
}

/*
 * Gets name from devclass id
 */
char *GetDevClassName(unsigned int id)
{
	aps_idev_class_t *pDev = devClassList;

	DebugEnter("GetDevClassName()");

	while((pDev != NULL) && (pDev->id != id)) {
		pDev = pDev->prev;
	} 

	DebugExit();
	return pDev ? pDev->dev_name : 0;
}

/*
 * 	Get id from device name
 */
unsigned int GetDevClassId(char *name)
{
	aps_idev_class_t *pDev = devClassList;

	DebugEnter("GetDevClassId()");

	while((pDev != NULL) && (strcmp(pDev->dev_name, name) != 0)) {
		pDev = pDev->prev;
	} 

	DebugExit();
	return pDev ? pDev->id : 0;
}
