/*
 * config.h - Daemon and Utility Interface Header
 *
 * Copyright (C) 1997-2000 SpellCaster Telecommunications Inc.
 * $Id: config.h,v 1.3 2004/08/09 01:13:09 bcrl Exp $
 * Released under the GNU Public License. See LICENSE file for details.
 */

#ifndef CONFIG_H
#define CONFIG_H

#include <sys/types.h>
#include <string.h>

typedef u_int8_t u8;
typedef u_int16_t u16;
typedef u_int32_t u32;
typedef u_int64_t u64;

/*
 * Fixed array lengths, use these and you can't go wrong ;)
 */
#define PORT_NAME_LEN		16
#define CLASS_NAME_LEN		16
#define USER_NAME_LEN		256
#define PASSWORD_LEN		256
#define SITE_NAME_LEN		256
#define PHONE_LEN		256

#define SYSNAME_LEN		32

#include "call.h"

/*
 * Policy masks indicate which fields are valid and which
 * should be ignored. When setting the policy using BIOC_SPOLICY you
 * must set the mask. Set the mask to indicate validity, clear it to
 * ignore.
 */
enum PolicyOpts {
	P_MAGIC,
	P_MRU,
	P_MRRU,
	P_PFC,
	P_ACFC,
	P_AUTH,
	P_ANSWER,
	P_ACCM,
	P_SSN,
	P_ECHO,
	P_LDISC,
	P_CALL_INFO,
	P_EPD_CLASS,
	P_EPD_LENGTH,
	NumPMASKs
};

#define PAUTH_PAP		0xC023
#define PAUTH_CHAP		0xC223

/*
 * Link-level policy and configuration structure
 */
class policy_t {
private:
	u32	data[NumPMASKs];

	/*
	 * validity mask for the policy. Indicates which of the following
	 * fields are to be ignored. Default is 0x00000000 (ignore everything)
	 */
	u32	valid_mask;

public:
	u8	epd_addr[20];

	void Clear(void) {
		valid_mask = 0;
		memset(data, 0, sizeof(data));
	}

	policy_t() {
		Clear();
	}

	int Valid(enum PolicyOpts opt) {
		return valid_mask & (1 << opt);
	}

	void Set(enum PolicyOpts opt, u32 value) {
		data[opt] = value;
		valid_mask |= 1 << opt;
	}

	u32 Get(enum PolicyOpts opt) {
		return data[opt];
	}

	void Clear(enum PolicyOpts opt) {
		valid_mask &= ~(1 << opt);
		data[opt] = 0;
	}

	void Set(policy_t &src) {
		*this = src;
	}

	void Merge(policy_t &src) {
		for (unsigned i=0; i<NumPMASKs; i++) {
			enum PolicyOpts opt = (enum PolicyOpts)i;
			if (src.Valid(opt))
				Set(opt, src.Get(opt));
		}
	}
};

/*
 * Policy Scope Codes indicate how a policy is to be applied
 */
#define PSCOPE_GLOBAL	0	/* Applies to all ports */
#define PSCOPE_CLASS	1	/* Applies to all ports of a single class */
#define PSCOPE_PORT	2	/* Applies to a single port */
#define PSCOPE_SESSION	3 	/* Applies for length of connection */

/*
 * Link-level policy setting
 */
class PolicyMsg_t {
public:
	unsigned char scope;
	char ident[PORT_NAME_LEN];
	policy_t policy;
};

/*
 * Configuration and authentication
 *
 * The OptionSet structure is passed between the BPE and the
 * config/authenticate daemon to exchange configuration information
 */
class OptionSet_t {
public:
	policy_t *pol;				/* policy for dial session (NULL if none) */
	Call	*call;			/* Call Identifier */

	u32	rem_ip;				/* IP address of remote peer */
	u32	loc_ip;				/* Local IP address of interface */
	u32	dns_primary;		/* Primary DNS server IP to assign peer */
	u32	dns_second;			/* Secondary DNS server IP to assign peer */
	u32	netmask;			/* Netmask to use for netroute above */

	u32	loc_fpeer;			/* Our BACP favored peer number */
	u32	rem_fpeer;			/* Peer's BACP favored peer number */
	int	addtime;			/* Seconds at HWM before adding */
	int	droptime;			/* Seconds at LWM before dropping */
	int	min_links, max_links,
		rx_drop_bpls,
		rx_raise_bpls,
		tx_drop_bpls,
		tx_raise_bpls;

	u32	call_info;			/* Additional call info */
	char	user[USER_NAME_LEN];		/* Auth username */
	char	passwd[PASSWORD_LEN];		/* PAP Auth password */
	char	site[SITE_NAME_LEN];		/* Remote sitename for dial-out */
	char	phone[PHONE_LEN];		/* Remote phone number to dial */
	char	dev_class[CLASS_NAME_LEN];	/* Device class to use */
	char	port[PORT_NAME_LEN];		/* Port to use */

	u8	chap_lvalue_len;
	u8	chap_lvalue[255];	/* Value we CHAP challenged the peer with */
	u8	chap_rvalue_len;
	u8	chap_rvalue[255];	/* Value we CHAP challenged the peer with */
	
	u32	is_valid : 1;		/* Configuration is valid */
	u32	vj : 1;				/* Negotiate/Use VJ IP Header Compression */
	u32	vjc : 1;			/* Negotiate/Use compressed VJ slot id */
	u32	netroute : 1;		/* Create a network route to peer */
	u32	proxy : 1;			/* Create proxy ARP entry for peer */
	u32	defroute : 1;		/* Create default route to peer */
	u32	reserved : 1;		/* Unused padding */
	u32	challenge : 1;		/* We are being challenged (CHAP only) */
	u8	auth_info;		/* CHAP identifier */
	u16	proto_id;		/* Authentication protocol ID */

	OptionSet_t() {
		memset(this, 0, sizeof(OptionSet_t));
	}
};

/* the start and stop message types MUST match rfc2866 Acct-Status-Type Value */
#define ACCT_START		1
#define ACCT_STOP		2

#define ACCT_TC_USERREQ	1	/* Remote termination */
#define ACCT_TC_NOCARR	2	/* Link Disconnection */
#define ACCT_TC_IDLE	4	/* Idle Timeout */
#define ACCT_TC_STIME	5	/* Session Timeout */
#define ACCT_TC_NONEED	12	/* BoD disconnect */
#define ACCT_TC_HOSTREQ	18	/* Local termination */

#define MAX_AC_MSGS		128	/* Queue no more than 16 accounting messages */
class AcctMessage_t {
public:
	u32		type;			/* ACCT_START or ACCT_STOP */
	u64		in_octets;
	u64		out_octets;
	u32		in_packets;
	u32		out_packets;
	u32		term_cause;
	Call		*call;

	char		port[PORT_NAME_LEN];		/* Name of port concerned */
	char		dev_class[CLASS_NAME_LEN];	/* Name of device class */
	char		user[USER_NAME_LEN];		/* Authentication username */
	char		ifname[16];			/* Interface name */

	char		reason[256];
};
	
#define MAX_RT_MSGS		128	/* Queue no more than 16 route messages */

#define ROUTE_NET		0
#define ROUTE_HOST		1
#define ROUTE_GATEWAY	2

class RouteMsg_t {
public:
	u32	dst_addr;
	u32	gw_addr;
	u32	mask;
	u16	flags : 2;
	u16	reserved : 14;
	u16	metric;
	char	dev[16];
};

#define MAX_PROXY_MSGS	128 /* Queue no more than 16 proxy messages */
class ProxyMsg_t {
public:
	u32	dst_addr;
};

#define MAX_CFG_MSGS	128	/* Queue no more than 16 config messages */
class CfgMessage_t {
public:
	void (*reserved1)(void *, OptionSet_t *);
	void *reserved2;
	OptionSet_t options;

	CfgMessage_t() {
		reserved1 = NULL;
		reserved2 = NULL;
	}
};

class HangupMsg_t {
public:
	unsigned int ident;			/* Unique call identifier */
	char port[PORT_NAME_LEN];	/* Port name */
};

class DialMsg_t {
public:
	unsigned int ident;
	policy_t *pol;
	char site[SITE_NAME_LEN];
	char phone[PHONE_LEN];
};

extern OptionSet_t *get_iface_options(void *iface);
extern void DialGotConfig(void *, OptionSet_t *options);
extern void babd_GetConfig(void (*cbf)(void *, OptionSet_t *), void *obj, OptionSet_t *options);

#endif
