/*
 * TTY Driver control program
 *
 * Copyright (C) 1997-2000 SpellCaster Telecommunications Inc.
 * $Id: ttyctrl.c,v 1.1.1.1 2004/03/11 03:59:31 bcrl Exp $
 * Released under the GNU Public License. See LICENSE file for details.
 */

#include <stdio.h>
#include <sys/ioctl.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <getopt.h>
#include <string.h>
#include <sys/param.h>
#include "ldisc.h"
#include "ttyd.h"
#include "version.h"

enum FLAGS {
	FLAG_RESET,
	FLAG_FORCE,
	FLAG_NEW,
	FLAG_CHECK,
	FLAG_KILL
};

char *arg0;

extern FILE *yyin;
extern struct linedef ttyconfig[MAXCONFIG];
extern struct optiondef options;
const char CONFIG_FILE[]="/etc/babylon/bttyd.conf";
char configFile[MAXPATHLEN];

#ifdef DEBUG
static char shrt_options[] = "h:vrnc:f:k";
#else
static char shrt_options[] = "h:vrnc:f:";
#endif

static struct option long_options[] = {
	{ "print_usage()", no_argument, NULL, 'f' },
	{ "version", no_argument, NULL, 'v' },
	{ "reset", no_argument, NULL, 'r' },
	{ "new-port", no_argument, NULL, 'n' },
	{ "check-config", optional_argument, NULL, 'c' },
	{ "force-idle", required_argument, NULL, 'f' },
#ifdef DEBUG
	{ "killdrv", optional_argument, NULL, 'k' },
#endif
};

int yyparse();

void print_usage() {
	fprintf(stderr, "Usage:  %s [options]\n", arg0);
	fprintf(stderr, "   -h, --help               display this info\n");
	fprintf(stderr, "   -c <s>                   check config specified in <s> file instead of %s\n",CONFIG_FILE);
	fprintf(stderr, "   --check-config <s>       check config, optionally use <s> file instead of %s\n",CONFIG_FILE);
	fprintf(stderr, "   -v, --version            display version and exit.\n");
	fprintf(stderr, "   -r, --reset              reset the TTY driver and trash all ports (Use with caution)\n");
	fprintf(stderr, "   -n, --new-port           setup a new Babylon port for the driver\n");
	fprintf(stderr, "   -f, --force-idle <s>     force a Babylon port idle (Use with caution!\n");
	exit(1);
}

int main ( int argc, char *argv[] ) {
	int port,flag,ctrl_fd;
	int opt,opt_indx;
	
	arg0=argv[0];
	port=-1;
	flag = 0;

	if (argc<2) {
			print_usage();
	}
	/* Process command line options */
	while((opt = getopt_long(argc, argv, shrt_options, long_options, &opt_indx)) != EOF) {
		switch(opt) {
				case 'v':
					printf("ttyctrl Version %s\n",version);
					exit(0);
				case 'r':
					if ((flag==FLAG_FORCE) || (flag==FLAG_NEW)) {
						puts("Can't use other command with force port idle");
						print_usage();
					}
					flag=FLAG_RESET;
					break;
				case 'f':
					if ((flag==FLAG_RESET) || (flag==FLAG_NEW)) {
						puts("Can't use other command with force port idle");
						print_usage();
					}
					if (optarg) {
							port=atoi(optarg);
					}
					flag=FLAG_FORCE;
					break;
				case 'n':
					if ((flag==FLAG_RESET) || (flag==FLAG_FORCE)) {
						puts("Can't use other command with new port");
						print_usage();
					}
					flag=FLAG_NEW;
					break;
				case 'c':
					/* Initialize configuration space */
					memset(&ttyconfig,sizeof(ttyconfig),(char)0);
					memset(&options,sizeof(options),(char)0);
					if (optarg) {
							strncpy(configFile,optarg,MAXPATHLEN);
					} else {
							strcpy(configFile,CONFIG_FILE);
					}

					/* Set our global filename item */
					if ((yyin=fopen(configFile,"r"))==NULL) {
						printf("Cannot open %s: %s\n",configFile,strerror(errno));
						exit(1);
					}
					/* Parse the config file */
					if (yyparse()>0) {
						puts("Error on config file");
					} else {
						puts("Config file OK");
					}
					fclose(yyin);
					exit(0);
#ifdef DEBUG
				case 'k':
					flag=FLAG_KILL;
					break;
#endif
				default:
					print_usage();
		}
	}
	if ((ctrl_fd = open("/proc/bttyctrl", O_RDONLY)) < 0) {
		perror("TTY: Cannot open /proc/bttyctrl");
		exit(1);
	}
	switch (flag) {
		case FLAG_RESET:
			if (ioctl(ctrl_fd,BIOCF_TTY_RESET,0)!=0) {
				perror("TTY: Driver reset failed");
				exit(1);
			} else {
				puts("TTY: Driver reset.");
			}
			break;
		case FLAG_FORCE:
			if (port==-1) {
				puts("TTY: Force command requires a port number");
				print_usage();
			}
			if (ioctl(ctrl_fd,BIOCF_TTY_STAT_IDLE,port)!=0) {
				perror("TTY: Set port idle failed");
				exit(1);
			} else {
				printf("TTY: Port %d set to status idle.\n",port);
			}
			break;
		case FLAG_NEW:
			if (ioctl(ctrl_fd,BIOCF_TTY_REGISTER_PORT,0)!=0) {
				perror("TTY: New port create failed");
				print_usage();
			} else {
				puts("TTY: Created new port");
			}
			break;
#ifdef DEBUG
		case FLAG_KILL:
			if (ioctl(ctrl_fd,BIOCF_TTY_KILL,0)!=0) {
				perror("TTY: Kill driver failed");
				print_usage();
			} else {
				puts("TTY: Command successful");
			}
			break;
			
#endif
	}
	close(ctrl_fd);
	return(0);
}


